/*
* Copyright (c) 2010 Michal Hruby <michal.mhr@gmail.com>
*               2022 elementary LLC. (https://elementary.io)
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
*
* You should have received a copy of the GNU General Public
* License along with this program; if not, write to the
* Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
* Boston, MA 02110-1301 USA
*
* Authored by: Michal Hruby <michal.mhr@gmail.com>
*/

namespace Synapse {
    public class CalculatorPlugin: Object, Activatable, ItemProvider {
        public bool enabled { get; set; default = true; }

        public void activate () { }
        public void deactivate () { }

        private class Result: Synapse.Match, Synapse.TextMatch {
            public int default_relevancy { get; set; default = 0; }

            public string text { get; construct set; default = ""; }
            public Synapse.TextOrigin text_origin { get; set; }

            public Result (string result, string match_string) {
                Object (match_type: MatchType.CALCULATION,
                        text: "%s".printf (result), //Copied to clipboard
                        title: "%s".printf (result), //Label for search item row
                        icon_name: "accessories-calculator",
                        text_origin: Synapse.TextOrigin.UNKNOWN
                );
            }
        }

        static void register_plugin () {
            DataSink.PluginRegistry.get_default ().register_plugin (
                typeof (CalculatorPlugin),
                "accessories-calculator",
                register_plugin,
                Environment.find_program_in_path ("bc") != null
            );
        }

        static construct {
            register_plugin ();
        }

        public bool handles_query (Query query) {
            return (QueryFlags.ACTIONS in query.query_type);
        }

        public async ResultSet? search (Query query) throws SearchError {
            ResultSet? results = null;
            try {
                string d = yield CalculatorPluginBackend.get_instance ().get_solution (
                    query.query_string,
                    query.cancellable
                ); // throws error if no valid solution found

                Result result = new Result (d, query.query_string);
                result.description = "%s\n%s".printf (
                    "%s = %s".printf (query.query_string, d),
                    Granite.TOOLTIP_SECONDARY_TEXT_MARKUP.printf (_("Click to copy result to clipboard"))
                );  // Used for search item tooltip

                results = new ResultSet ();

                results.add (result, Match.Score.AVERAGE);
            } catch (Error e) {
                if (!(e is IOError.FAILED_HANDLED)) {
                    warning ("Error processing %s with bc: %s", query.query_string, e.message);
                }
            }

            query.check_cancellable ();
            return results;
        }
    }
}
