package ro.sync.lexer.c;
import ro.sync.lexer.AbstractLexer;

@SuppressWarnings("unused")
%%

%public 
%class CLexer
%extends AbstractLexer
%unicode
%char
%type ro.sync.lexer.Symbol
 
%ignorecase
%scanerror ro.sync.lexer.LexerException


%{
    private static final byte SYM_TEXT               = CTokens.TEXT;
    private static final byte SYM_KEYWORD            = CTokens.KEYWORD;
    private static final byte SYM_IDENTIFIER         = CTokens.IDENTIFIER;
    private static final byte SYM_OPERATOR           = CTokens.OPERATOR;
    private static final byte SYM_STRING             = CTokens.STRING;
    private static final byte SYM_CHAR_LITERAL       = CTokens.CHAR_LITERAL;
    private static final byte SYM_NUMBER             = CTokens.NUMBER;
    
    private static final byte SYM_CURLY_BRACKET      = CTokens.CURLY_BRACKET;
    private static final byte SYM_SQUARE_BRACKET     = CTokens.SQUARE_BRACKET;
    private static final byte SYM_BRACKET            = CTokens.BRACKET;
    private static final byte SYM_COMMA              = CTokens.COMMA;
    private static final byte SYM_COLON              = CTokens.COLON;
    private static final byte SYM_SEMICOLON          = CTokens.SEMICOLON;
    private static final byte SYM_DOT                = CTokens.DOT;
    private static final byte SYM_POINTER_REFERENCE  = CTokens.DOT;
    private static final byte SYM_THREE_DOTS         = CTokens.THREE_DOTS;
    
    private static final byte SYM_COMMENT            = CTokens.COMMENT;
    private static final byte SYM_LINE_COMMAND       = CTokens.LINE_COMMAND;
    
    
    /**
     * Create an empty lexer, yyreset will be called later to reset and assign
     * the reader
     */
    public CLexer() {
        super();
    }
    
    public String getName() {
      return C_LEXER;
    }
%}

%xstate COMMENT, DQ_STRING

Keyword = "if" | "else" | "switch" | "while" | "do" | "for" | "goto" | "continue" | "break"
        | "return" | "extern" | "static" | "auto" | "register" | "void" | "char" | "short" | "int" 
        | "long" | "float" | "double" | "signed" | "unsigned" | "struct" | "union" | "enum" 
        | "const" | "volatile" | "typedef" | "sizeof" | "case" | "default" | "null"

Operator = "=" | "*" | "+" | "++" | "-" | "--" | "/" | "%" | "&" | "&&" | "~" | "!" | "*=" | "/="
         | "%=" | "+=" | "-=" | "<<=" | ">>=" | "&=" | "^=" | "|=" | "?" | "|" | "||" | "^" | "=="
         | "!=" | "<" | ">" | "<=" | ">=" | "<<" | ">>"

// Identifiers
IdentifierStart = [a-zA-Z$_]
Identifier = {IdentifierStart} ({IdentifierStart} | [0-9])* 
 
// Numbers
IntegerType = ([uU]? [lL]) | ([uU] [lL]?)

Digit = [0-9]
Decimal = {Digit}+ {IntegerType}?

HexDigit = [0-9a-fA-F]
Hexadecimal = "0" [xX] {HexDigit}+ {IntegerType}+

Exponent = [eE] [+-]? {Digit}+
FloatType = [fFdD]
FloatingPoint = ({Digit}+ "." {Digit}* {Exponent}? {FloatType}?)
              | ("." {Digit}+ {Exponent}? {FloatType}?)
              | ({Digit}+ {Exponent} {FloatType}?)
              | ({Digit}+ {Exponent}? {FloatType})

Number = {Decimal} | {Hexadecimal} | {FloatingPoint}

// Escape sequences
OctalEscape = "\\" [0-7]+
UnicodeEscape = "\\u" {HexDigit}+
EscapeSequence = ("\\" [btnfr\"\'\\]) | {OctalEscape} | {UnicodeEscape}

// Character literals
UnclosedCharLiteral = "\'" ({EscapeSequence} | [^\'\\])
CharLiteral         = {UnclosedCharLiteral} "\'" 

// Line comments
LineComment = "//" {Char}*

// Line commands
LineCommand = "#" {Char}*

// Whitespaces
WS = [ \t]

// Any character not handled separatelly.
GeneralChar = [^;,=:\{\}\(\)\[\]*.+\-/%<>&~!\^|?\"\'\t ]

// Any character
Char = .

%%

<YYINITIAL> {
    // White spaces are emitted separatelly.
    {WS}+                        {   return symbol(SYM_TEXT);           }
	
    {Keyword}                    {   return symbol(SYM_KEYWORD);        }
    {Identifier}                 {   return symbol(SYM_IDENTIFIER);     }
    {Operator}                   {   return symbol(SYM_OPERATOR);       }
    "{" | "}"                    {   return symbol(SYM_CURLY_BRACKET);  }
    "[" | "]"                    {   return symbol(SYM_SQUARE_BRACKET); }
    "(" | ")"                    {   return symbol(SYM_BRACKET);        }
    ";"                          {   return symbol(SYM_SEMICOLON);      }
    ":"                          {   return symbol(SYM_COLON);          }
    "."                          {   return symbol(SYM_DOT);            }
    "->"                         {   return symbol(SYM_POINTER_REFERENCE); }
    "..."                        {   return symbol(SYM_THREE_DOTS);     }
    ","                          {   return symbol(SYM_COMMA);          }   
    "\""                         {   
                                     cLen++;
                                     yybegin(DQ_STRING);         
                                 }
    {CharLiteral} 
    | {UnclosedCharLiteral}      {   return symbol(SYM_CHAR_LITERAL);   }
    {Number}                     {   return symbol(SYM_NUMBER);         }
    {LineCommand}                {   return symbol(SYM_LINE_COMMAND);   }
    {LineComment}                {   return symbol(SYM_COMMENT);        }
    "/*"                         {
                                     yybegin(COMMENT);
                                     return symbol(SYM_COMMENT);
                                 }
    
	// This is Text
    {GeneralChar}+               {	 return symbol(SYM_TEXT);           }
}

<DQ_STRING> {
    "\""                         {
                                     cLen++;
                                     yybegin(YYINITIAL);
                                     return flush(SYM_STRING);
                                 }
    "\\\\" | "\\\""              {   cLen+=2;                           }
    [^\"]                        {   cLen++;                            }
    <<EOF>>                      {
                                     yybegin(YYINITIAL);
                                     return flush(SYM_STRING);
                                 }
}

<COMMENT> {
  "*/"                           {
                                     yybegin(YYINITIAL);
                                     return symbol(SYM_COMMENT);                                     
                                 }
  {Char}                         {   cLen ++;   }                                 
  ~"*/"                          {
                                     yypushback(2);
                                     return symbol(SYM_COMMENT);
                                 }
  <<EOF>>                        {   return flush(SYM_COMMENT);    }
}

