package ro.sync.lexer.json;
import ro.sync.lexer.AbstractLexer;
import ro.sync.lexer.LexicalState;
import ro.sync.lexer.LexicalStateUnlimittedStack;

@SuppressWarnings("unused")
%%

%public 
%class JSONLexer
%extends AbstractLexer
%unicode
%char
%type ro.sync.lexer.Symbol
 
%scanerror ro.sync.lexer.LexerException

%{
    // Operators and stuff.
    private static final byte SYM_COMMA = JSONTokens.COMMA;
    private static final byte SYM_SQUARE_BRACKET = JSONTokens.SQUARE_BRACKET;
    private static final byte SYM_CURLY_BRACKET = JSONTokens.CURLY_BRACKET;
    private static final byte SYM_COLON = JSONTokens.COLON;
    
    // Most keywords.
    private static final byte SYM_BOOLEAN = JSONTokens.BOOLEAN;
    private static final byte SYM_NULL = JSONTokens.NULL;
    private static final byte SYM_NUMBER = JSONTokens.NUMBER;
    
    // Property
    private static final byte SYM_PROPERTY = JSONTokens.PROPERTY;
    // Value
    private static final byte SYM_VALUE = JSONTokens.VALUE;
    
    // Invalid content
    private static final byte SYM_INVALID = JSONTokens.INVALID;
    // Emit spaces separatelly.
    private static final byte SYM_TEXT = JSONTokens.TEXT;
    
    /**
     * Create an empty lexer, yyreset will be called later to reset and assign
     * the reader
     */
    public JSONLexer() {
        super();
    }
    
    public String getName() {
      return JSON_LEXER;
    }
    
    /**
     * Create a new lexical state to be used during tokenization.
     * It is supposed to be equivalent with YYINITIAL...
     * 
     * @return  The lexical state.
     */
    protected LexicalState createLexicalState() {
      return new LexicalStateUnlimittedStack();
    }
    
    private static final LexicalState INITIAL_STATE = new LexicalStateUnlimittedStack();
    
    /**
     * Create a new lexical state to be used during tokenization.
     * It is supposed to be equivalent with YYINITIAL...
     * 
     * @return  The lexical state.
     */
    protected LexicalState getInitialLexicalState() {
      return INITIAL_STATE;
    }
%}

%xstate PROPERTY, VALUE, ARRAY_VALUE

DQStringContent =  ([^\"]|\\\")*
SQStringContent =  ([^\']|\\\')*

DQUnclosedString =  \"{DQStringContent}
SQUnclosedString =  \'{SQStringContent}

DQString =  {DQUnclosedString}\"
SQString =  {SQUnclosedString}\'

// true, false...
Boolean = "true" | "false"

// null...
Null = "null"

Digit = [0-9]
Integer = "-"?{Digit}+
Fraction = "."{Digit}+
Exponent = ("e" | "E") ("+" | "-")? {Digit}+
Double = {Integer}({Fraction} {Exponent}? | {Exponent})

Number = {Integer} | {Double}
             
GeneralChar = [^,\"\[\]\{\} \t:\.]

%%

<YYINITIAL> {
    "["                         {   
                                    yybegin(ARRAY_VALUE);
                                    return symbol(SYM_SQUARE_BRACKET);
                                }
    "]"                         {   
                                    return symbol(SYM_SQUARE_BRACKET);
                                }
    "{"                         {
                                    pushState(YYINITIAL);
                                    yybegin(PROPERTY);
                                    return symbol(SYM_CURLY_BRACKET);
                                }
    // White spaces are emitted separatelly.
    [ \t]+                      {   return symbol(SYM_TEXT);                }
    // White spaces are emitted separatelly.
    {GeneralChar}*              {   return symbol(SYM_INVALID);             }
}

<PROPERTY> {
    // Keywords.
    {Null}                      {   return symbol(SYM_NULL);                }
    // Strings
    {DQString}                  {   return symbol(SYM_PROPERTY);            }
    {SQString}                  {   return symbol(SYM_PROPERTY);            }
    // White spaces are emitted separatelly.
    [ \t]+                      {   return symbol(SYM_TEXT);                }
    ":"                         {
                                    yybegin(VALUE);
                                    return symbol(SYM_COLON);
                                }
     "{"                         {
                                    pushState(PROPERTY);
                                    yybegin(PROPERTY);
                                    return symbol(SYM_CURLY_BRACKET);
                                }
    "}"                         {
                                    yybegin(popState());
                                    return symbol(SYM_CURLY_BRACKET);
                                }
    ","                         {   return symbol(SYM_COMMA);               }
    // Match anything else different from the markup.
    {GeneralChar}*              {   return symbol(SYM_INVALID);             }
}

<VALUE> {
    // Keywords.
    {Boolean}                   {   return symbol(SYM_BOOLEAN);             }
    {Number}                    {   return symbol(SYM_NUMBER);              }
    {Null}                      {   return symbol(SYM_NULL);                }
    // Strings.
    {DQString}                  {   return symbol(SYM_VALUE);               }
    {SQString}                  {   return symbol(SYM_VALUE);               }
    // Operators and punctuation marks.
    ","                         {
                                    yybegin(PROPERTY);
                                    return symbol(SYM_COMMA);
                                }
    "["                         {
                                    pushState(VALUE);
                                    yybegin(ARRAY_VALUE);
                                    return symbol(SYM_SQUARE_BRACKET);      }
    "{"                         {
                                    pushState(VALUE);
                                    yybegin(PROPERTY);
                                    return symbol(SYM_CURLY_BRACKET);
                                }
    "}"                         {
                                    yybegin(popState());
                                    return symbol(SYM_CURLY_BRACKET);
                                }
    // White spaces are emitted separatelly.
    [ \t]+                      {   return symbol(SYM_TEXT);                }
    // Match anything else different from the markup.
    {GeneralChar}*              {   return symbol(SYM_INVALID);             }
}

<ARRAY_VALUE> {
    // Keywords.
    {Boolean}                   {   return symbol(SYM_BOOLEAN);             }
    {Number}                    {   return symbol(SYM_NUMBER);              }
    {Null}                      {   return symbol(SYM_NULL);                }
    // Strings
    {DQString}                  {   return symbol(SYM_VALUE);               }
    {SQString}                  {   return symbol(SYM_VALUE);               }
    // Operators and punctuation marks.
    ","                         {   return symbol(SYM_COMMA);               }
    "["                         {
                                    pushState(ARRAY_VALUE);
                                    return symbol(SYM_SQUARE_BRACKET);
                                }
    "]"                         {
                                    yybegin(popState());
                                    return symbol(SYM_SQUARE_BRACKET);
                                }
    "{"                         {
                                    pushState(ARRAY_VALUE);
                                    yybegin(PROPERTY);
                                    return symbol(SYM_CURLY_BRACKET);
                                }
    "}"                         {
                                    yybegin(popState());
                                    return symbol(SYM_CURLY_BRACKET);
                                }
    // White spaces are emitted separatelly.
    [ \t]+                      {   return symbol(SYM_TEXT);                }
    // Match anything else different from the markup.
    {GeneralChar}*              {   return symbol(SYM_INVALID);             }
}