package ro.sync.lexer.rnc;
import ro.sync.lexer.AbstractLexer;

@SuppressWarnings("unused")
%%

%public 
%class RNCLexer
%extends AbstractLexer
%unicode
%char
%type ro.sync.lexer.Symbol
 
%scanerror ro.sync.lexer.LexerException

%{
    // Operators and stuff.
    private static final byte SYM_SLASH = RNCTokens.OPERATOR;
    private static final byte SYM_BACKSLASH = RNCTokens.OPERATOR;
    private static final byte SYM_COMMA = RNCTokens.OPERATOR;
    private static final byte SYM_DOT = RNCTokens.OPERATOR;
    private static final byte SYM_ASTERISK = RNCTokens.OPERATOR;
    private static final byte SYM_CURLY_BRACKET = RNCTokens.CURLY_BRACKET;
    private static final byte SYM_BRACKET = RNCTokens.BRACKET;
    private static final byte SYM_QUESTION = RNCTokens.OPERATOR;
    private static final byte SYM_COLON = RNCTokens.OPERATOR;
    private static final byte SYM_COLON_STAR = RNCTokens.OPERATOR;
    private static final byte SYM_AMPERSAND = RNCTokens.OPERATOR;
    private static final byte SYM_PLUS = RNCTokens.OPERATOR;
    private static final byte SYM_MINUS = RNCTokens.OPERATOR;
    private static final byte SYM_EQ = RNCTokens.OPERATOR;
    private static final byte SYM_PIPE = RNCTokens.OPERATOR;
    private static final byte SYM_AMPERSANDEQ = RNCTokens.OPERATOR;
    private static final byte SYM_PIPEEQ = RNCTokens.OPERATOR;
    // Most keywords.
    private static final byte SYM_KEYWORD1 = RNCTokens.KEYWORD1;
    private static final byte SYM_KEYWORD2 = RNCTokens.KEYWORD2;
    private static final byte SYM_STRING_DQ = RNCTokens.STRING_DQ;
    private static final byte SYM_STRING_SQ = RNCTokens.STRING_SQ;
    // Comments
    private static final byte SYM_COMMENT1 = RNCTokens.COMMENT1;
    private static final byte SYM_COMMENT2 = RNCTokens.COMMENT2;
    private static final byte SYM_ANNOTATION = RNCTokens.ANNOTATION;
    // Other text.
    private static final byte SYM_TEXT = RNCTokens.TEXT;
    
    public byte annotationLevel = 0;
    
    /**
     * Create an empty lexer, yyreset will be called later to reset and assign
     * the reader
     */
    public RNCLexer() {
        super();
    }
    
    public String getName() {
      return RNC_LEXER;
    }
%}

%xstate SQ_STRING, DQ_STRING, COMMENT1, COMMENT2, ANNOTATION

DQStringContent =  ([^\"]|\\\")*
SQStringContent =  ([^\']|\\\')*

DQUnclosedString =  \"{DQStringContent}
SQUnclosedString =  \'{SQStringContent}

DQString =  {DQUnclosedString}\"
SQString =  {SQUnclosedString}\'

Keyword1 = "namespace" | "default" | "datatypes" | "inherit"
             
Keyword2 = "element" | "attribute" | "list" | "mixed" | "parent" | "empty" | "text" | "notAllowed"
            | "external" | "grammar" | "start" | "div" | "include" | "string" | "token"

GeneralChar = [^\\,:{}\-+|\'\"\[\] \t=*\.#\(\)?]

%%

<YYINITIAL> {
    // Keywords.
    {Keyword1}                  {   return symbol(SYM_KEYWORD1);            }
    {Keyword2}                  {   return symbol(SYM_KEYWORD2);            }
    // Operators and punctuation marks.
    "="                         {   return symbol(SYM_EQ);                  }
    "|="                        {   return symbol(SYM_PIPEEQ);              }
    "&="                        {   return symbol(SYM_AMPERSANDEQ);         }
    ","                         {   return symbol(SYM_COMMA);               }
    "&"                         {   return symbol(SYM_AMPERSAND);           }
    "|"                         {   return symbol(SYM_PIPE);                }
    "{" | "}"                   {   return symbol(SYM_CURLY_BRACKET);       }
    "(" | ")"                   {   return symbol(SYM_BRACKET);             }
    "+"                         {   return symbol(SYM_PLUS);                }
    "-"                         {   return symbol(SYM_MINUS);               }
    "?"                         {   return symbol(SYM_QUESTION);            }
    "*"                         {   return symbol(SYM_ASTERISK);            }
    "."                         {   return symbol(SYM_DOT);                 }
    ":"                         {   return symbol(SYM_COLON);               }
    ":*"                        {   return symbol(SYM_COLON_STAR);          }
    "\\"                        {   return symbol(SYM_BACKSLASH);           }
    
    // Comments
    "#"                         {
                                    // Save state to return to.
                                    pushState(YYINITIAL);
                                    yybegin(COMMENT1);
                                    // Reset the char counter
                                    cLen = 0;
                                    return symbol(SYM_COMMENT1);  
                                }
    "##"                        {
                                    // Save state to return to.
                                    pushState(YYINITIAL);
                                    yybegin(COMMENT2);
                                    // Reset the char counter
                                    cLen = 0;
                                    return symbol(SYM_COMMENT2);  
                                }
    "["                         {
                                    // Save state to return to.
                                    pushState(YYINITIAL);
                                    yybegin(ANNOTATION);
                                    // Reset the char counter
                                    cLen = 0;
                                    return symbol(SYM_ANNOTATION);  
                                }
    // Strings
    {DQString}                  {   return symbol(SYM_STRING_DQ);           }
    {SQString}                  {   return symbol(SYM_STRING_SQ);           }
    "/"                         {   return symbol(SYM_SLASH);               }
    // White spaces are emitted separatelly.
    [ \t]+                      {   return symbol(SYM_TEXT);                }
    // This is Text
    // Match anything else different from the markup.
    {GeneralChar}*              {   return symbol(SYM_TEXT);                }
    <<EOF>>                     {   return flush(SYM_TEXT);                 }
}

<DQ_STRING> {
    "\""                        {
                                    yybegin(YYINITIAL);
                                    cLen++;
                                    return flush(SYM_STRING_DQ);
                                }
    .                           {   cLen++;    }
    "\\\""                      {   cLen += 2; }
    <<EOF>>                     {   return flush(SYM_STRING_DQ);            }
}

<SQ_STRING> {
    "'"                         {
                                    yybegin(YYINITIAL);
                                    cLen++;
                                    return flush(SYM_STRING_SQ);
                                }
    .                           {   cLen++;                                 }
    "\\\'"                      {   cLen += 2;                              }
    <<EOF>>                     {   return flush(SYM_STRING_SQ);            }
}

<COMMENT1> {
    .                           {   cLen++;                                 }
    <<EOF>>                     {
                                    yybegin(popState());
                                    return flush(SYM_COMMENT1);    
                                }
}

<COMMENT2> {
    .                           {   cLen++;                                 }
    <<EOF>>                     {
                                    yybegin(popState());
                                    return flush(SYM_COMMENT2);    
                                }
}

<ANNOTATION> {
    "["                         {   
                                    if (cLen > 0) {
                                        yypushback(1);
                                        return flush(SYM_ANNOTATION);
                                    } else {
                                        annotationLevel++;
                                        return symbol(SYM_ANNOTATION);
                                    }
                                }
    "]"                         {   
                                    if (cLen > 0) {
                                        yypushback(1);
                                        return flush(SYM_ANNOTATION);
                                    } else {
                                        annotationLevel--;
                                        if (annotationLevel <= 0) {
                                            yybegin(popState());
                                        }
                                        return symbol(SYM_ANNOTATION);
                                    }
                                }
    .                           {   cLen++;                                 }
    <<EOF>>                     {   return flush(SYM_ANNOTATION);           }
}