package ro.sync.lexer.xml;
import ro.sync.lexer.AbstractLexer;

@SuppressWarnings("unused")
%%

%public 
%class XMLLexer
%extends AbstractLexer
%unicode
%char
%type ro.sync.lexer.Symbol
%scanerror ro.sync.lexer.LexerException

%{
    public static final byte SYM_TEXT               = XMLTokens.TEXT;
    public static final byte SYM_INVALID            = XMLTokens.INVALID;
    public static final byte SYM_ENTITY             = XMLTokens.ENTITY;

    public static final byte SYM_AUTO_CLOSING_TAG   = XMLTokens.XML_TAG;
    public static final byte SYM_TAG_START          = XMLTokens.XML_TAG;
    public static final byte SYM_OPEN_TAG_CLOSE     = XMLTokens.XML_TAG;      
    public static final byte SYM_OPEN_TAG_END       = XMLTokens.XML_TAG;
    public static final byte SYM_OPEN_START_TAG     = XMLTokens.XML_TAG;
    public static final byte SYM_CLOSE_TAG          = XMLTokens.XML_TAG;
    public static final byte SYM_LT                 = XMLTokens.XML_TAG;
    public static final byte SYM_SLASH              = XMLTokens.XML_TAG;

    public static final byte SYM_INSTR_START        = XMLTokens.PI;
    public static final byte SYM_INSTR              = XMLTokens.PI;
    public static final byte SYM_INSTR_END          = XMLTokens.PI;

    public static final byte SYM_XML_HEADER         = XMLTokens.XML_HEADER;

    public static final byte SYM_COMMENT_START      = XMLTokens.COMMENT;
    public static final byte SYM_COMMENT            = XMLTokens.COMMENT;
    public static final byte SYM_COMMENT_END        = XMLTokens.COMMENT;
    
    public static final byte SYM_CDATA              = XMLTokens.CDATA;

    public static final byte SYM_JSP                = XMLTokens.PI;
    
    public static final byte SYM_ATTRIBUTE_NAME     = XMLTokens.ATTRIBUTE_NAME;
    public static final byte SYM_EQ                 = XMLTokens.EQ;
    public static final byte SYM_ATTRIBUTE_VALUE_DQ = XMLTokens.ATTRIBUTE_VALUE_DQ;
    public static final byte SYM_ATTRIBUTE_VALUE_SQ = XMLTokens.ATTRIBUTE_VALUE_SQ;
        
    public static final byte SYM_DOCTYPE_INNER      = XMLTokens.DOCTYPE_INNER;
    public static final byte SYM_DOCTYPE_INNER_COMMENT   = XMLTokens.DOCTYPE_INNER;
    public static final byte SYM_DOCTYPE_STRING_DQ = XMLTokens.DOCTYPE_INNER;
    public static final byte SYM_DOCTYPE_STRING_SQ = XMLTokens.DOCTYPE_INNER;

    public static final byte SYM_DOCTYPE           = XMLTokens.DOCTYPE;
    
    /**
     * If <code>true</code> the Lexer will handle special cases for PHP.
     * See EXM-29528
     */
    private boolean forPHP = false;
    
    /**
     * Create an empty lexer, yyreset will be called later to reset and assign
     * the reader.
     *
     * forPHP If <code>true</code> the lexer will be used for "pure" php files.
     */
    public XMLLexer(boolean forPHP) {
        super();
        this.forPHP = forPHP;
    }
    
    public String getName() {
      return XML_LEXER;
    }
%}

%state COMMENT

%xstate CDATA, TAG, AFTER_ATTRIBUTE, AFTER_EQUAL, END_TAG, END_TAG_AFTER_NEW_LINE
%xstate ATTRIBUTE_DQ_STRING, ATTRIBUTE_SQ_STRING
%xstate INSTR, PHP_INSTR, JSP, XML_INSTR

%xstate DOCTYPE, DOCTYPE_DQ_STRING, DOCTYPE_SQ_STRING
%xstate DOCTYPE_INTERNAL_SUBSET, DOCTYPE_INTERNAL_SUBSET_COMMENT, DOCTYPE_INTERNAL_SUBSET_SQ_STRING
%xstate DOCTYPE_INTERNAL_SUBSET_DQ_STRING

/* main character classes */

/* white space - checked against W3C*/
S = (\u0020 | \u0009 | \u000D | \u000A | \u3000)+

/* characters */

/* any Unicode character, excluding the surrogate blocks, FFFE, and FFFF  - checked ok against W3C */
Char = \u0009 | \u000A | \u000D | [\u0020-\uD7FF] | [\uE000-\uFFFD] | [\u10000-\u10FFFF]
        | [\u20000–\u2FFFD] | [\u30000–\u3FFFD] | [\u40000–\u4FFFD] | [\u50000–\u5FFFD]
        | [\u60000–\u6FFFD] | [\u70000–\u7FFFD] | [\u80000–\u8FFFD] | [\u90000–\u9FFFD]
        | [\uA0000–\uAFFFD] | [\uB0000–\uBFFFD] | [\uC0000–\uCFFFD] | [\uD0000–\uDFFFD]
        | [\uE0000–\uEFFFD] | [\uF0000–\uFFFFD] | [\u100000–\u10FFFD]

NameStartChar = [A-Z] | "_" | [a-z] | 
                    [\u00C0-\u00D6] | 
                    [\u00D8-\u00F6] | 
                    [\u00F8-\u02FF] | 
                    [\u0370-\u037D] | 
                    [\u037F-\u1FFF] | 
                    [\u200C-\u200D] | 
                    
                    [\u2070-\u218F] | 
                    [\u2C00-\u2FEF] | 
                    [\u3001-\uD7FF] | 
                    [\uF900-\uFDCF] | 
                    [\uFDF0-\uFFFD] 
  
// This is mapped by the compiled lexer to a rule that matches everything.                  
//                    | 
//                    [\u10000-\uEFFFF]
                    
NameChar = {NameStartChar} | "-" | "." | [0-9] | \u00B7 | [\u0300-\u036F] | [\u203F-\u2040]
Name = {NameChar}+

QName = {Name} | ({Name} ":" {Name})
RelaxedQName = {Name} | ({Name}? ":" {Name}?)

/* Tags */
/* Compatibility - do not accept space between name and end, let other production match it.*/
AutoClosingTag = "<" {RelaxedQName} "/>"

TagStart = ("<" {RelaxedQName} ">") | "<>"

OpenTagStart = "<" {RelaxedQName}

OpenTagEnd = "</" {RelaxedQName}

CloseTag = ("</" {RelaxedQName} ">") | "</>"

Entity = "&" "#"? {NameChar}* ";"

BadEntity = "&" "#"? {NameChar}* ("<" | ">")?

PI_DQStringContent =  ([^\"]|\\\")*
PI_SQStringContent =  ([^\']|\\\')*

PI_DQUnclosedString =  \"{PI_DQStringContent}
PI_SQUnclosedString =  \'{PI_SQStringContent}

PI_DQString =  {PI_DQUnclosedString}\"
PI_SQString =  {PI_SQUnclosedString}\'

/* attribute */
Attribute = {QName}
%%

<YYINITIAL> {
  
  {Entity}                       {   return symbol(SYM_ENTITY);         }

  // EXM-22847
  {BadEntity}                    {   
                                     String tokenText = yytext();
                                     char ch = tokenText.charAt(tokenText.length() - 1);
                                     if (ch == '<' || ch == '>') {
                                       yypushback(1);
                                     }
                                     return symbol(SYM_INVALID); 
                                 }

  "<?xml "                       {
                                     // The XML processing instruction, we treat it specially.
                                     yybegin(XML_INSTR);
                                     cLen = 0;
                                     yypushback(6);
                                 }
  "<?php"                        {
                                     // Push back the php word, it will be part of the PI body !
                                     yypushback(3);
                                     yybegin(forPHP ? PHP_INSTR : INSTR);
                                     return symbol(SYM_INSTR_START);
                                 }
  "<?"		                       {
                                     yybegin(INSTR);
                                     return symbol(SYM_INSTR_START);
                                 }
  {AutoClosingTag}               {
                                     return symbol(SYM_AUTO_CLOSING_TAG);
                                 }                               
  {TagStart}                     {
                                     return symbol(SYM_TAG_START);
                                 }
  {OpenTagStart}                 {
                                     yybegin(TAG);
                                     return symbol(SYM_OPEN_START_TAG);
                                 }
  {CloseTag}                     {   
                                     return symbol(SYM_CLOSE_TAG); 
                                 }
  {OpenTagEnd}                   {
                                     yybegin(END_TAG);
                                     return symbol(SYM_CLOSE_TAG);
                                 }
  "</"                    		   {   
                                     return symbol(SYM_CLOSE_TAG); 
                                 }
  "<!--"                         {
                                     yybegin(COMMENT);
                                     // Reset the char counter
                                     cLen = 0;
                                     return symbol(SYM_COMMENT_START);
                                 }
  "<![CDATA["                    {
                                     yybegin(CDATA);
                                     cLen = 0;
                                     // Reset the char counter
                                     yypushback(9);
                                   
                                 }
  // Compatibility stuff. Match this as error like the previous TokenMarker did.                             
  "<!"                    		   {   
                                     return symbol(SYM_INVALID); 
                                 }
  "<%"                           {
                                     yybegin(JSP);
                                     // Compatibility - We will include the start into the token.
                                     yypushback(2);
                                     // Reset the char counter                                     
                                     cLen = 0;
                                 }
  "<!DOCTYPE"                    {
                                     yybegin(DOCTYPE);
                                     // Compatibility - We will include the start into the token.
                                     cLen = 9;
                                 }
  "<"                    		     {   
                                     yybegin(TAG);
                                     return symbol(SYM_LT); 
                                 }
  [^<&]*                         {
                                     // This is Text
                                     // Match anything else different from the markup.
                                     return symbol(SYM_TEXT);
                                 }
}

<TAG> {
  ({S}* {Name} ":")              {
                                    yybegin(AFTER_ATTRIBUTE);
                                    // Compatibility - stop at the :  
                                    return symbol(SYM_ATTRIBUTE_NAME); 
                                 }
  ({S}* ":" {Name})              {
                                    yybegin(AFTER_ATTRIBUTE);
                                    // Compatibility - accept the : at the begining...  
                                    return symbol(SYM_ATTRIBUTE_NAME); 
                                 }
  ({S}* {Attribute} {S}*)        {
                                    yybegin(AFTER_ATTRIBUTE);
                                    // Compatibility - agreagate the space into the attribute name.  
                                    return symbol(SYM_ATTRIBUTE_NAME); 
                                 }
  // This might be a tag start so try to recover from here.                               
  "<"						                 {
                  									 // Compatibility. Recognize this as a start tag.
                  									 yypushback(1);
                  									 yybegin(YYINITIAL);  									
                                 }
  "="                            {   
                                     yypushback(1);
                                     yybegin(AFTER_ATTRIBUTE);
                                 }  
  \"                             {
                                     yybegin(ATTRIBUTE_DQ_STRING);
                                     // Compatibility - let the quote to enter the attribute value.
                                     // Reset the char counter, but count the quote
                                     cLen = 1;                                     
                                 }
  \'                             {
                                     yybegin(ATTRIBUTE_SQ_STRING);
                                     // Compatibility - let the quote to enter the attribute value.
                                     // Reset the char counter, but count the quote
                                     cLen = 1;                                     
                                 }
  ("/>")                         {
                                     yybegin(YYINITIAL);
                                     return symbol(SYM_OPEN_TAG_CLOSE);
                                 }
  ("/")                          {   
  									 // Maybe it is an incomplete tag close. 
                                     return symbol(SYM_SLASH);
                                 }
  {Entity}                       {   return symbol(SYM_ENTITY);         }
  // EXM-22847
  {BadEntity}                    {   
                                     String tokenText = yytext();
                                     char ch = tokenText.charAt(tokenText.length() - 1);
                                     if (ch == '<' || ch == '>') {
                                       yypushback(1);
                                     }
                                     return symbol(SYM_INVALID); 
                                 }
  ">"                            {
                                     // Compatibility - inject the space into the token.
                                     yybegin(YYINITIAL);
                                     return symbol(SYM_OPEN_TAG_END);
                                 }
  {S}*                           {
                                     // Compatibility - this can be the remainder space between 
                                     // the last attribute and the end part of the tag                                    
                                     return symbol(SYM_ATTRIBUTE_NAME);
                                 }
  //([^< \t]+ "<"?)                   {
    //                                 if (yytext().endsWith("<")) {
      //                                 yypushback(1);
        //                             }
          //                           return symbol(SYM_INVALID);
            //                     }
}

<AFTER_ATTRIBUTE> {
  ({S}+ "=")                     {
                                     // Compatibility - this can be the remainder space between 
                                     // the last attribute and the end part of the tag                                    
                                     yypushback(1);
                                     return symbol(SYM_ATTRIBUTE_NAME);
                                 }
  ("=" {S}*)                     {
                                     yybegin(AFTER_EQUAL);
                                     return symbol(SYM_EQ);
                                 }
  // This might be a tag start so try to recover from here.                               
  "<"                            {
                                     // Compatibility. Recognize this as a start tag.
                                     yypushback(1);
                                     yybegin(YYINITIAL);                                    
                                 }
  ("/>")                         {
                                     yybegin(YYINITIAL);
                                     return symbol(SYM_OPEN_TAG_CLOSE);
                                 }
  ("/")                          {   
                                     // Maybe it is an incomplete tag close. 
                                     return symbol(SYM_SLASH);
                                 }
  {Entity}                       {   return symbol(SYM_ENTITY);         }
  // EXM-22847
  {BadEntity}                    {   
                                     String tokenText = yytext();
                                     char ch = tokenText.charAt(tokenText.length() - 1);
                                     if (ch == '<' || ch == '>') {
                                       yypushback(1);
                                     }
                                     return symbol(SYM_INVALID); 
                                 }
  ">"                            {
                                     // Compatibility - inject the space into the token.
                                     yybegin(YYINITIAL);
                                     return symbol(SYM_OPEN_TAG_END);
                                 }
  [^<>/&=]+                      {
                                    return symbol(SYM_TEXT);
                                 }
}


<AFTER_EQUAL> {
  // This might be a tag start so try to recover from here.                               
  "<"                            {
                                     // Compatibility. Recognize this as a start tag.
                                     yypushback(1);
                                     yybegin(YYINITIAL);                                    
                                 }
  \"                             {
                                     yybegin(ATTRIBUTE_DQ_STRING);
                                     // Compatibility - let the quote to enter the attribute value.
                                     // Reset the char counter, but count the quote
                                     cLen = 1;                                     
                                 }
  \'                             {
                                     yybegin(ATTRIBUTE_SQ_STRING);
                                     // Compatibility - let the quote to enter the attribute value.
                                     // Reset the char counter, but count the quote
                                     cLen = 1;                                     
                                 }
  ("/>")                         {
                                     yybegin(YYINITIAL);
                                     return symbol(SYM_OPEN_TAG_CLOSE);
                                 }
  ("/")                          {   
                                     // Maybe it is an incomplete tag close. 
                                     return symbol(SYM_SLASH);
                                 }
  {Entity}                       {   return symbol(SYM_ENTITY);         }
  // EXM-22847
  {BadEntity}                    {   
                                     String tokenText = yytext();
                                     char ch = tokenText.charAt(tokenText.length() - 1);
                                     if (ch == '<' || ch == '>') {
                                       yypushback(1);
                                     }
                                     return symbol(SYM_INVALID); 
                                 }
  ">"                            {
                                     // Compatibility - inject the space into the token.
                                     yybegin(YYINITIAL);
                                     return symbol(SYM_OPEN_TAG_END);
                                 }
  {S}*                           {
                                     // Compatibility - this can be the remainder space between 
                                     // the last attribute and the end part of the tag                                    
                                     return symbol(SYM_ATTRIBUTE_NAME);
                                 }
  ({S}* {Name} ":")              {
                                    return symbol(SYM_TEXT); 
                                 }
  ({S}* ":" {Name})              {
                                    return symbol(SYM_TEXT); 
                                 }
  ({S}* {Attribute} {S}*)        {
                                    return symbol(SYM_TEXT); 
                                 }
}

<END_TAG> {
  // This might be a tag start so try to recover from here.                               
  "<"						     {
  									// Compatibility. Recognize this as a start tag.
  									yypushback(1);
  									yybegin(YYINITIAL);  									
                                 }                                 
  {S}*                           {
                                     // Compatibility - this can be the remainder space between 
                                     // the tag name and the end part of the tag                                    
                                     return symbol(SYM_ATTRIBUTE_NAME);
                                 }                                 
  (">")                          {
                                     yybegin(YYINITIAL);
                                     return symbol(SYM_CLOSE_TAG);
                                 }
  <<EOF>>                        {
                                     yybegin(END_TAG_AFTER_NEW_LINE);
                                 }
}

<END_TAG_AFTER_NEW_LINE> {
  // This might be a tag start so try to recover from here.                               
  "<"                            {
                                    // Compatibility. Recognize this as a start tag.
                                    yypushback(1);
                                    yybegin(YYINITIAL);                                     
                                 }                                 
  {S}*                           {
                                     // Compatibility - this can be the remainder space between 
                                     // the tag name and the end part of the tag                                    
                                     return symbol(SYM_ATTRIBUTE_NAME);
                                 }                                 
  (">")                          {
                                     yybegin(YYINITIAL);
                                     return symbol(SYM_CLOSE_TAG);
                                 }
}


/* Should be equivalent to ATTRIBUTE_SQ_STRING, but different tokens. */
<ATTRIBUTE_DQ_STRING> {
  \"                            {
                                    cLen ++;                                                                        
                                    // Back to tag state
                                    yybegin(TAG);                                                                        
                                    // Flush accumulated buffer.                                    
                                    return flush(SYM_ATTRIBUTE_VALUE_DQ);                                    
                                }
  // This might be a tag start so try to recover from here.                                
  "<"						                {
                  					  	    // Compatibility. Recognize this as a start tag.
                  									yypushback(1);
                  									yybegin(YYINITIAL); 
                  									return flush(SYM_ATTRIBUTE_VALUE_DQ); 									
                                }         
  [^\"]                         {   cLen ++;  }                               
  <<EOF>>                       {   return flush(SYM_ATTRIBUTE_VALUE_DQ);  }
}

<ATTRIBUTE_SQ_STRING> {
  \'                            {
                                    cLen ++;                                                                        
                                    // Back to tag state
                                    yybegin(TAG);                                                                                                           
                                    return flush(SYM_ATTRIBUTE_VALUE_SQ);                                    
                                }
  // This might be a tag start so try to recover from here.
  "<"						                {
                								    // Compatibility. Recognize this as a start tag.
                  									yypushback(1);
                  									yybegin(YYINITIAL); 
                  									return flush(SYM_ATTRIBUTE_VALUE_SQ); 									
                                } 
  [^\']                         {   cLen ++;  }                               
  <<EOF>>                       {   return flush(SYM_ATTRIBUTE_VALUE_SQ);  }
}


/* Similar to JSP, includes start and end into the token. */
<CDATA> {
  [^\]]*						 {
  								
  									cLen += yylength();
  								 }    								
  "]"                            {
                                     cLen ++;
                                 }                                 
  ~"]]>"                         {
                                     // Do not push back. We are trying to be as compatible as 
                                     // possible to the current token marker. We'll consume it in 
                                     // the current token.                                         
                                     yybegin(YYINITIAL);                                                            
                                     return symbol(SYM_CDATA);
                                 }
  <<EOF>>                        {        
                                     return flush(SYM_CDATA);
                                 }
}

/* Similar to CDATA,  includes start and end into the token.*/
<JSP> {
  [^%]*							 {
  									cLen += yylength();
  								 }    								
  "%"                            {
                                     cLen ++;
                                 }                                 
  ~"%>"                      	 {
                                     // Do not push back. We are trying to be as compatible as 
                                     // possible to the current token marker. We'll consume it in 
                                     // the current token.  
                                     // yypushback(2);     
                                     yybegin(YYINITIAL);                                                            
                                     return symbol(SYM_JSP);
                                 }
  <<EOF>>                        {        
                                     return flush(SYM_JSP);
                                 }
}

<COMMENT> {
  [^-]*                          {
                                     cLen += yylength();
                                 }                    
  "-"							 {
  									 cLen ++;
  							     }                                           
  "-->"                          {
                                     yybegin(YYINITIAL);
                                     return symbol(SYM_COMMENT_END);                                     
                                 }
  ~"-->"                         {
                                     yypushback(3);
                                     return symbol(SYM_COMMENT);
                                 }
  <<EOF>>                        {
                                    return flush(SYM_COMMENT);
                                 }
}

/* Should be equivalent to XML_INSTR, but different tokens, and end token handling. */
<PHP_INSTR> {  
  [^?\"\']*                      {
                                     cLen += yylength();
                                 }
  {PI_SQString} |
  {PI_DQString} |
  {PI_SQUnclosedString} |
  {PI_DQUnclosedString}          {
                                     cLen += yylength();
                                 }
  "?"                            {
                                     cLen ++;
                                 }
  "?>"                           {
                                     if (cLen > 0) {
                                        yypushback(2);
                                        return flush(SYM_INSTR);
                                     } else {
                                        yybegin(YYINITIAL);
                                        return symbol(SYM_INSTR_END);
                                     }                                     
                                 }
  <<EOF>>                        {   return flush(SYM_INSTR);   }
}

/* Should be equivalent to XML_INSTR, but different tokens, and end token handling. */
<INSTR> {  
  [^?]*                          {
                                     cLen += yylength();
                                 }                    
  "?"							 {
  									 cLen ++;
  							     }                                           
  "?>"                     		 {                                     
                                     yybegin(YYINITIAL);
                                     return symbol(SYM_INSTR_END);                                     
                                 }
  ~"?>"                    	     {
                                     // Compatible to the old token marker. The end of the 
                                     // PI is returned as a separate token.
                                     yypushback(2);                            
          				             return symbol(SYM_INSTR);
                                 }
  <<EOF>>                        {   return flush(SYM_INSTR);   }
}
/* Similar to JSP, CDATA, includes start and end into the token.*/
<XML_INSTR> {
  [^?]*                          {
                                     cLen += yylength();
                                 }                    
  "?"							 {
  									 cLen ++;
  							     }                         
  ~"?>"                    		 {
                                     // Do not push back. We are trying to be as compatible as 
                                     // possible to the current token marker. We'll consume it in 
                                     // the current token.  
                                     // yypushback(2);     
                                     yybegin(YYINITIAL);                                                            
                                     return symbol(SYM_XML_HEADER);
                                 }
  <<EOF>>                        {   return flush(SYM_XML_HEADER);    }
}


/* In doctype we may have strings, or an internal subset */
<DOCTYPE> {
  /* End of DOCTYPE */
  ">"                           {
                                    yybegin(YYINITIAL);
                                    cLen ++;
                                    return flush(SYM_DOCTYPE);
                                 }  
  /* Internal subset */                                 
  "["                            {
                                    cLen ++;
                                    yybegin(DOCTYPE_INTERNAL_SUBSET);
                                    return flush(SYM_DOCTYPE);
                                 }
  /* Public or system identifier values */                                                                     
  \"                             {
                                     yybegin(DOCTYPE_DQ_STRING); 
                                     cLen ++;
                                 }
  \'                             {
                                     yybegin(DOCTYPE_SQ_STRING); 
                                     cLen ++;
                                 }
  // This might be a tag start so try to recover from here.
  "<"  						     {
                  								   // Compatibility. Recognize this as a start tag.
                  									 yypushback(1);
                  									 yybegin(YYINITIAL); 
                  									 return flush(SYM_DOCTYPE); 									
                                 } 
                                 
  {Char}                         {   cLen ++;   }                                 
  <<EOF>>                        {   return flush(SYM_DOCTYPE);  }                                
}

/* Should be equivalent to DOCTYPE_SQ_STRING, but different tokens. */
<DOCTYPE_DQ_STRING> {
  \"                            {
                                    cLen ++;                                                                        
                                    yybegin(DOCTYPE);                                                                        
                                }
  {Char}                        {   cLen ++;  }                               
  <<EOF>>                       {   return flush(SYM_DOCTYPE);  }
}
<DOCTYPE_SQ_STRING> {
  \'                            {
                                    cLen ++;            
                                    yybegin(DOCTYPE);                                                                                                           
                                }
  {Char}                        {   cLen ++;  }                               
  <<EOF>>                       {
                                    return flush(SYM_DOCTYPE);
                                }
}

<DOCTYPE_INTERNAL_SUBSET> {
    "]"                         {   
                                    if (cLen > 0) {
                                        // Push back the square bracket...
                                        yypushback(1);
                                        return flush(SYM_DOCTYPE_INNER);
                                    } else {
                                        // Compatibility - the end marker is emitted as SYM_DOCTYPE
                                        yybegin(DOCTYPE);
                                        return symbol(SYM_DOCTYPE);
                                    }
                                }
    "<!--"                      {
                                    cLen += 4;
                                    yybegin(DOCTYPE_INTERNAL_SUBSET_COMMENT);
                                }
    \" | 
    \'                          {
                                    if(cLen > 0){
                                        // We have an accumulated buffer. Flush it first,
                                        // Then expect the quote in the same state.
                                        yypushback(1);
                                        return flush(SYM_DOCTYPE_INNER);
                                    } else {
                                        // No buffer
	                                    // Change state
	                                    if(yycharat(0) == '"'){                                    
	                                       yybegin(DOCTYPE_INTERNAL_SUBSET_DQ_STRING);
	                                    } else {
                                           yybegin(DOCTYPE_INTERNAL_SUBSET_SQ_STRING);	                                    
	                                    }                                    
                                        return symbol(SYM_DOCTYPE_INNER);
                                    }
                                    
                                 }                                
                                
    {Char}                       {   cLen ++;  }                               
    <<EOF>>                      {   return flush(SYM_DOCTYPE_INNER);  }
}

<DOCTYPE_INTERNAL_SUBSET_COMMENT> {
  "-->"                          {
                                     yybegin(DOCTYPE_INTERNAL_SUBSET);
                                     // Add the comment end length.
                                     cLen +=3;
                                 }
  {Char}                         {   cLen ++;   }                                 
  <<EOF>>                        {   return flush(SYM_DOCTYPE_INNER);   }
}

/* String with tokens for the start and end.*/
<DOCTYPE_INTERNAL_SUBSET_SQ_STRING> {
  \'                             {                                     
                                     yybegin(DOCTYPE_INTERNAL_SUBSET);
                                     return symbol(SYM_DOCTYPE_INNER);                                     
                                 }
  {Char}                         {   cLen ++;   }                                 
  ~\'                            {
                                     yypushback(1);     
                                     return symbol(SYM_DOCTYPE_INNER);
                                 }
  <<EOF>>                        {   return flush(SYM_DOCTYPE_INNER);  }
}

<DOCTYPE_INTERNAL_SUBSET_DQ_STRING> {
  \"                             {                                     
                                     yybegin(DOCTYPE_INTERNAL_SUBSET);
                                     return symbol(SYM_DOCTYPE_INNER);                                     
                                 }
  {Char}                         {   cLen ++;  }                                 
  ~\"                            {
                                     yypushback(1);     
                                     return symbol(SYM_DOCTYPE_INNER);
                                 }
  <<EOF>>                        {   return flush(SYM_DOCTYPE_INNER);  }
}